/*  TIMPROC.C  Time complete est-tune and retune process using dummy
 *  speech data.  Retune BEFORE tuning estimator for test purposes.
 *  Compile with AAFILT.C, AARETUN.C, CCOR.C, INCHIST.C,
 *  and SRCHIST.C.
 */

#include <stdio.h>
#include <math.h>

void retune(int rem,float shf,int n,float in[],float out[]);
void ccor(int n,float in[],float win[],float xr[],float xi[]);
void est(int n,float xr[],float xi[],float smf,
	 float *pk,float *ptch,float *shf);
void inchist(int nb,float bs[],float hptch,float hsft,float incr);
void srchist(int nb,float bs[],float thr,int *mwid,float *ctr);

#define SAMP_RATE  8000.0 /*  Nominal sampling rate in Hz     */
#define FFT_N       512   /*  FFT and IFFT block length       */
#define BLKS_TO_EST   8   /*  Num input blks per mistune est  */
#define LOW_HIST_F -800.0 /*  Low limit histogram frequency   */
#define HI_HIST_F   800.0 /*  Highest hist bin bottom freq    */
#define HIST_N     1600   /*  Number of histogram bins        */
#define SUM_PCT      80.0 /*  Percent of sum of histogram increments
			   *  to be in min width histogram interval
			   */
#define HZTOBINS (HIST_N)/((HI_HIST_F)-(LOW_HIST_F))  /*  Bins per Hz  */
#define RETUNE_N ((int)(SAMP_RATE)/(BLKS_TO_EST)) /* Retune blocklen */

main ()
{
   static float in[RETUNE_N], retout[RETUNE_N];
   /*  in = admittedly nonsensical input data
    *  retout = retuner output data
    */
   static float win[FFT_N], fftr[FFT_N], ffti[FFT_N];
   /*  win = raised-cosine data window
    *  fftr = fft and complex correlation real part
    *  ffti =                             imag part
    */
   static float histbins[HIST_N];
   /*  The bins of the power versus frequency histogram
    */
   float act_shift, ccorppow, pitch, estshift, suminc;
   /*  act_shift = Actual sim mistuning freq shift in sim Hz
    *  ccorppow = height of ccor (interpolated) peak mag-squared
    *  pitch = value of ccor estimated speech pitch in Hz
    *  estshift = ccor est of (one possible) freq shift in Hz
    *  suminc = sum of hist increments since last his reset
    */
   int idx, jdx, kdx, itau, nmloop, mloopx, minwid;
   /*  minwid = num-1 hist bins in min width interval w sum >= thresh
    */
   float ctr, shift_hz, f_delt;
   /*  ctr = est in binwidths of mistuning minus LOW_HIST_F
    *  shift_hz = ctr converted to Hz minus zero frequency
    *  f_delt = 0.5 * (minwid+1) * (bin width in Hz)
    */

   for(idx=0; idx<FFT_N; idx++)
   {  /*  Define raised-cosine window  */
      win[idx]=1.0-cos(2.0*3.1415926*(idx+0.5)/(FFT_N));
   }
   nmloop=1;
   while(nmloop>0)
   {
      printf("FFT pts %d, hist bin %g Hz wide, range %g to %g.\n",
	      FFT_N,         1./(HZTOBINS), LOW_HIST_F,HI_HIST_F);
      printf("Min width hist intvl %g pct incr fm reset.\n",SUM_PCT);
      printf("Enter retune freq Hz (samp rate %g Hz).\n",SAMP_RATE);
      scanf("%f",&act_shift);
      printf("Enter number of times thru (nom 1 sec) loop.\n");
      scanf("%d",&nmloop);
      for(mloopx=0;mloopx<nmloop;mloopx++)
      {
	 for(idx=0;idx<HIST_N;idx++)
	 {  /*  Clear histogram  */
	    histbins[idx]=0.;
	 }
	 suminc=0.;  /*  Reset sum of histogram increments  */
	 for(idx=0;idx<BLKS_TO_EST;idx++)
	 {  /*  Vary dummy rep period (tau) from low to high  */
	    itau=40 + 11*idx;  /*  tau increment shd be prime  */
	    for(jdx=0;jdx<RETUNE_N;jdx+=itau)
	    {  /*  sawtooth sims speech  */
	       for(kdx=0;kdx<itau && jdx+kdx<RETUNE_N;kdx++)
	       {  in[jdx+kdx]=(1-(2.*kdx+1)/(float)itau);
	       }
	    }

	    retune(0, act_shift/(SAMP_RATE), RETUNE_N, in, retout);
	    /*  Skip first 64 points of retuner output  */
	    ccor(FFT_N, &retout[64], win, fftr, ffti);
	    est(FFT_N,fftr, ffti, SAMP_RATE,
		&ccorppow, &pitch, &estshift);
	    suminc+=ccorppow;
	    inchist(HIST_N, histbins,
		    pitch*HZTOBINS, (estshift-LOW_HIST_F)*HZTOBINS,
		    ccorppow);
	 }

	 srchist(HIST_N, histbins, (SUM_PCT*0.01)*suminc,
		 &minwid, &ctr);
	 shift_hz=LOW_HIST_F+(ctr+0.5)/(HZTOBINS);
	 f_delt=(minwid+1.0)/(2.0*HZTOBINS);  /* minwid==0 for 1 bin */
      }
      printf("Done.\n");
      printf("Est shift %g Hz, +/- %g.\n",shift_hz,f_delt);
   }
}

