/*  AAFILT.C  */
#include <stdio.h>
/*#define DEBUG*/
#define NFILTS 7  /* Number of filters in filter bank */
#define NTAPS 21  /* Number of taps in each fir filter */

float fir(float in, float *tr_coef, int n, float *history);

void aafilt(int remember, float rel_sft, int n_pts, float in[],
            float out[])
/*
 *  Anti-aliasing fir filters routine.  Selects filter from filter bank,
 *  and zeros filter memory, when remember==0.
 *
 *  remember==0:  Determine filter, zero its memory, proceed.
 *  remember!=0:  Use previously determined filter and its memory
 *  rel_sft = Proposed frequency shift divided by sampling frequency
 *  n_pts = Number of data points to anti-alias filter
 *  in = n_pts data vector in
 *  out                    out
 */
{
   static float filtmem[NTAPS-1];  /* filter init cond-memory */
   static int ifilt;  /* index to selected filter */
   /* these are static so can be remembered between calls */

   int idx;

   static float aadat[NFILTS][NTAPS+1] = {
   {-.12,  /* hipass 0.16 rpl 0.163, stop 0.12 max gain 0.054 */
   0.0348, -.0712, -.0334, 0.0024, 0.0401, 0.0585, 0.0321,
   -.0461, -.1540, -.2478, 0.7150, -.2478, -.1540, -.0461,
   0.0321, 0.0585, 0.0401, 0.0024, -.0334, -.0712, 0.0348},

   {-.08,  /* hipass 0.12 rpl 0.160, stop 0.08 max gain 0.053 */
   -.0613, 0.0386, 0.0432, 0.0450, 0.0345, 0.0060, -.0400,
   -.0963, -.1507, -.1902, 0.7954, -.1902, -.1507, -.0963,
   -.0400, 0.0060, 0.0345, 0.0450, 0.0432, 0.0386, -.0613},
   
   {-.04,  /* hipass 0.08 rpl 0.150, stop 0.04 max gain 0.050 */
   0.0773, 0.0017, -.0067, -.0202, -.0379, -.0581, -.0790,
   -.0982, -.1139, -.1241, 0.8724, -.1241, -.1139, -.0982,
   -.0790, -.0581, -.0379, -.0202, -.0067, 0.0017, 0.0773},

   {0.04,  /* bandpass 0.04 to 0.46 rpl 0.160, stops 0.01 & 0.49 */                                              
   -.1105, 0.,     -.0686, 0.,     -.0832, 0.,     -.0950,
   0.,     -.1027, 0.,     0.8946, 0.,     -.1027, 0.,
   -.0950, 0.,     -.0832, 0.,     -.0686, 0.,     -.1105},

   {0.08,  /* lopass 0.42 rpl 0.150, stop 0.46 max gain 0.050 */
   0.0772, -.0017, -.0066, 0.0202, -.0379, 0.0581, -.0790,
   0.0983, -.1139, 0.1240, 0.8724, 0.1240, -.1139, 0.0983,
   -.0790, 0.0581, -.0379, 0.0202, -.0066, -.0017, 0.0772},

   {0.12,  /* lopass 0.38 rpl 0.160, stop 0.42 max gain 0.053 */
   -.0613, -.0387, 0.0431, -.0450, 0.0346, -.0060, -.0401,
   0.0963, -.1507, 0.1902, 0.7954, 0.1902, -.1507, 0.0963,
   -.0401, -.0060, 0.0346, -.0450, 0.0431, -.0387, -.0613},
   
   {0.16,  /* lopass 0.34 rpl 0.163, stop 0.38 max gain 0.054 */
   0.0348, 0.0712, -.0334, -.0024, 0.0401, -.0585, 0.0321,
   0.0461, -.1541, 0.2478, 0.7151, 0.2478, -.1541, 0.0461,
   0.0321, -.0585, 0.0401, -.0024, -.0334, 0.0712, 0.0348}};

   if(remember==0)
   {  for(idx=0; idx<NTAPS-1; idx++)
      {  filtmem[idx]=0.;
      }
      for(ifilt=0; ifilt<NFILTS-1 && aadat[ifilt][0]<rel_sft; ifilt++);
      /* want range 0 <= ifilt < NFILTS */
   
   }

#ifdef DEBUG
   printf("Ifilt %d, relcut %g.\n",ifilt,aadat[ifilt][0]);
#endif

   for(idx=0; idx<n_pts; idx++)
   {  out[idx]=fir(in[idx], &aadat[ifilt][1], NTAPS, filtmem);
   }
}

float fir(float in, float *tr_coef, int n, float *history)
/*
 *  finite impulse response (fir) filter computation,
 *  adapted from Embree listing 4.1.
 *
 *  in = one floating-point input
 *  tr_coef = time-reversed or time-symmetric impulse response
 *  n = number of terms of tr_coef.  Must be odd and >=5.
 *  history = n-1 size array of past inputs and/or initial conditions
 *
 *  returns one floating-point output
 */
{
   int i;
   float *hist_ptr, *hist1_ptr, *tr_coef_ptr;
   float out;

   hist_ptr=history;
   hist1_ptr=hist_ptr;  /* use for history update */
   tr_coef_ptr=tr_coef;  /* fir coefs must be symmetric or time reversed */

   /* accumulate output */
   out=*hist_ptr++ * (*tr_coef_ptr++); /* 1st contribution */
   *hist1_ptr++ = *hist_ptr;  /* update a term of history array */
   out += (*hist_ptr++) * (*tr_coef_ptr++);  /* 2nd contribution */
   for(i=3; i<n; i+=2)
   {  *hist1_ptr++ = *hist_ptr;
      out += (*hist_ptr++)*(*tr_coef_ptr++);  /* ith contribution */
      *hist1_ptr++ = *hist_ptr;
      out += (*hist_ptr++)*(*tr_coef_ptr++);  /* (i+1)st contribution */
   }
   *hist1_ptr=in;  /* put input into history */
   out += in * (*tr_coef_ptr);  /* nth contribution */

   return(out);
}
