/*

  CDE/HyGain Keypad Entry Rotator Controller
  
  The following table is for the RobotDyn 4x4 Analog Keypad

  Run the Analog Keypad Calibration sketch to create the Keypad Mapping table if needed

  The default mapping for the RobotDyn 4x4 Analog Keypad
  const int keypad_map[] = {1023,931,853,787,675,633,596,563,503,480,458,438,401,322,269,231};

  Uses Adafruit ADS1015 Analog to Digital Converter library
  Uses Adafruit ST7735 and GFX TFT display libraries
  Uses Wire library to communicate with I2C A/D module
  Uses SPI library to communicate with SPI TFT display

*/

//#define debug   // Enables diagnostic information
//#define debug1  // Keypad related diagnostic information
//#define debug2 // Rotator Positioning and A/D diagnostic information
//#define debug3 // Individual A/D read values
//#define debug4  // Rotation calculation and movement diagnostic information

#include <Wire.h> //I2C Library
#include <Adafruit_ADS1015.h> // ADS1x15 A/D Library
#include <Adafruit_GFX.h>    // Core graphics library
#include <Adafruit_ST7735.h> // Hardware-specific library
#include <SPI.h>  // SPI library so we can communicate with the TFT display

#define default_cal_0 150  // The default ADC Zero (-180 degrees) value
#define default_cal_max 26895   // The default ADC Max (+180 degrees) value


// Map the calibrated analog keypad values to an array
const int keypad_map[] = {1023, 931, 853, 787, 675, 633, 596, 563, 503, 480, 458, 438, 401, 322, 269, 231};
// Map the keypad keys to their matching text
const String key_text[] = {"1", "2", "3", "A", "4", "5", "6", "B", "7", "8", "9", "C", "D",  "0", "E", "F"};
const String numbers = "1234567890";  // Creates a string used to compare if a pressed key is a number

#define comm_speed 115200 // Define the Arduino Serial Monitor baud rate
#define keypad_pin  A0  // Define the keypad analog input pin
#define no_key_pressed 0  // The A/D value for when no key is pressed
#define samples 5 // The number of times to read the key when pressed
#define read_delay 10  // Delay between keypad data samples
#define variance 10 // The A/D count tolerance for the calibrated key value versus the actual value
#define update_time 1000  // The display update time in milliseconds
#define degree_tolerance 1  // The rotator error tolerance in degrees
#define brake_time 3000 // Define the Brake delay in milliseconds
#define release_delay 500 // Define the delay between the brake release and rotation start
#define left_rotate_trim -3 // Define the rotate error fine tuning value for left (counterclockwise) rotation
#define right_rotate_trim 3 // Define the rotate error fine tuning value for right (clockwise) rotation

#define TFT_CS     10  // Assign the TFT CS to pin 10
#define TFT_RST    7  // Assign the TFT RST to pin 7
#define TFT_DC     8  // Assign the TFT DC to pin 6
#define tft_delay 10  // set the TFT command delay to 10ms

#define left_relay 4  // Define the I/O pin for the rotate left relay
#define right_relay 5 // Define the I/O pin for the rotate right relay
#define brake_relay 6 // Define the I/O pin for the brake release relay

int key_value;  // Variable to hold the averaged A/D value of the pressed key
int analog_value; // Variable to hold the converted analog value of the pressed key
int zero_point = no_key_pressed + variance;
int key_index =  -1;  // The index point to the location of the pressed key in the keypad_map array
String test_key;  // Variable to hold the text value of the pressed key
String move_to = "";  // Variable for the text of the entered bearing
bool moving = false;  // Flag to indicate if the rotator is turning
bool data_changed = true; // Flag to indicate that the display data has changed
int entered_bearing = -1; // Variable to hold the entered destination
int current_bearing;  // Variable to hold the current bearing
int previous_bearing = -1;  // Variable to hold the previous bearing
int actual_bearing; // Variable to hold the raw bearing value (-180 to +180)
boolean first_pass = true;  // Variable used to determine if first pass through the main loop
int adc_value;  // Variable to hold the ADS1115 A/D Value
int adc_cal_0 = default_cal_0;  // Variable to hold the ADS1115 ADC zero calibration point
int adc_cal_max = default_cal_max;  // Variable to hold the ADS1115 ADC max calibration point
int destination_bearing;  // Variable to hold the destination bearing
String rotate_direction;  // Variable to indicate if the rotation is to the left (counterclockwise) or right (clockwise)
unsigned long next_update_time = 0; // Variable to hold the next display update time in millis()

Adafruit_ADS1115 ads;  // Create an instance for the ADS1115 16-bit A/D

Adafruit_ST7735 tft = Adafruit_ST7735(TFT_CS,  TFT_DC, TFT_RST);  // Create the TFT display instance

void setup()
{
  pinMode(left_relay, OUTPUT);  // Set the rotate left relay pin to output
  pinMode(right_relay, OUTPUT);  // Set the rotate right relay pin to output
  pinMode(brake_relay, OUTPUT);  // Set the brake release relay pin to output

  digitalWrite(left_relay, LOW);  // Turn off the rotate left relay
  digitalWrite(right_relay, LOW);  // Turn off the rotate right relay
  digitalWrite(brake_relay, LOW);  // Turn off the brake release relay

#ifdef debug
  Serial.begin(comm_speed); // Set the Serial Monitor port speed
  Serial.println("Starting");
#endif

  ads.setGain(GAIN_ONE);        // Set the ADC Gain  to 1x gain   +/- 4.096V  1 bit = 0.125mV

  ads.begin();// Start the A/D Converter

  tft.initR(INITR_18BLACKTAB);   // initialize a 1.8" TFT with ST7735S chip, black tab
  delay(tft_delay);
  tft.fillScreen(ST7735_BLACK); // Clear the display - fill with BLACK background
  delay(tft_delay);
  tft.setRotation(1); // Set the screen rotation
  delay(tft_delay);
  tft.setTextWrap(false); // Turn off Text Wrap
  delay(tft_delay);
  tft.setTextSize(3); // Set the Font Size
  delay(tft_delay);
  tft.setTextColor(ST7735_BLUE); //Set the Text Color
  delay(tft_delay);
  tft.setCursor(40, 10);  //Set the Cursor and display the startup screen
  delay(tft_delay);
  tft.print("KW5GP");
  delay(tft_delay);
  tft.setTextSize(2);
  delay(tft_delay);
  tft.setCursor(10, 60);
  delay(tft_delay);
  tft.print("Keypad Entry");
  delay(tft_delay);
  tft.setCursor(45, 80);
  delay(tft_delay);
  tft.print("Rotator");
  delay(tft_delay);
  tft.setCursor(25, 100);
  delay(tft_delay);
  tft.print("Controller" );

  delay(5000);  //Wait 5 seconds then clear the startup message
  tft.fillScreen(ST7735_BLACK); // Clear the display - set to use a black background
  delay(tft_delay);

  read_adc(); // Read the ADC to get the initial values

  update_display(); // update the display

} // End of Setup Loop

void loop()
{
  read_key(); // Check the keypad to see if a key is pressed
  if ((millis() > next_update_time && data_changed) || first_pass)  // Check to see if we need to update the TFT display
  {
    update_display(); // Update the data on the TFT display
  }

  if (key_index != -1)  // Check to see if there is a valid key pressed
  {
    test_key = key_text[key_index]; // Get the text for the pressed key
    // We have a valid key

#ifdef debug1
    Serial.print("Valid Key: ");
    Serial.println(key_index);
#endif

    // Key has been decoded - determine action
    //If it's a number, check that's it's a valid number
    // Key codes 0-2, 4-6, 8-10 and 13 are digit entries

#ifdef debug1
    Serial.print("Key Text: ");
    Serial.print(test_key);
    Serial.print("   Checking numbers string: ");
    Serial.println(numbers.indexOf(test_key));
#endif

    if (numbers.indexOf(test_key) != -1)  // Check to see if the test key is in the numbers string (it's a number key)
    {
      if (!moving)  // Allow bearing entry when not moving
      {
        // It's a valid number - add it to direction string

        move_to = move_to + key_text[key_index];  // Add the key entry to the desired bearing
        entered_bearing = move_to.toInt();  // Convert the desired bearing to a string

#ifdef debug
        Serial.print("Valid Number: ");
        Serial.println(key_text[key_index]);
        Serial.print("Bearing: ");
        Serial.println(entered_bearing);
#endif

        if (entered_bearing >= 360)  // Clear the entered bearing if it's over 359
        {
#ifdef debug
          Serial.println("Entry Error - Bearing Cleared");
#endif
          move_to = "";
          entered_bearing = -1;
        }
        data_changed = true;  // Flag to indicate the display needs to be updated
      }
    } else
    {
      // It's a command key
#ifdef debug
      Serial.println("Command Key Pressed");
#endif
      if (test_key == "E")  //Clear Bearing Entry
      {
        if (!moving)  // Don't execute if moving
        {
          move_to = ""; // Reset the bearing entry string
          entered_bearing = -1; // Reset the entered bearing
          data_changed = true;// Flag to indicate the display needs to be updated

#ifdef debug
          Serial.println("Clear Key Pressed - Bearing cleared");
#endif
        }
      } else
      {
        if (test_key == "D")  // Go Key
        {
          if (entered_bearing != -1 && !moving) // Only execute if not moving and a valid destination has been entered
          {

#ifdef debug
            Serial.print("Go Key Pressed - Destination Bearing: ");
            Serial.println(entered_bearing);
#endif

            start_rotate(); // Process the rotation

          }
        } else
        {
          if (test_key == "F")  // Stop Key
          {

#ifdef debug
            Serial.print("Stop Key Pressed");
#endif

            if (moving) // If rotating, stop rotation
            {
#ifdef debug
              Serial.println(" - Rotation stopped");
#endif

              stop_all(); // Stop the rotation
            }
            if (entered_bearing != -1)  // Clear the bearing entry if it's a valid destination
            {
#ifdef debug
              Serial.println(" - Bearing cleared");
#endif
            }
            move_to = ""; //  Reset the bearing string
            entered_bearing = -1; // Reset the bearing entry
            data_changed = true;  // Indicate the display needs updating
          }
        }
      }
    }
  }

  read_adc(); // Read the ADS1115 A/D

  if (moving) // If we're moving - check where we are and stop if we're at the destination
  {
    check_move(); // Calls the function to process the rotation
  }
} // End of Main Loop

void read_key() // read_key function - Reads the 4x4 analog keypad
{
  int average = 0;  // Local variable to hold the average analog value
  int count = samples;  // Local variable to hold the number of data samples to get
  key_value = 0;  // Clear the incoming key value
  key_index = -1; // Clear the keypad_map array pointer

  if (analogRead(keypad_pin) > zero_point)  // Valid Key Pressed - otherwise exit
  {
    for (count = 0; count < samples + 1 ; count++)  // Get the desired number of keypad data samples
    {
      analog_value = analogRead(keypad_pin);  // Read the keypad analog input pin
      delay (read_delay); // Delay between samplings
      if (count != 0) // Only if the number of requested samples is not zero
      {
        average = average + analog_value; // Add the analog value to the average
      }
      key_value = average / samples;  // Calcluate the average value
    }

#ifdef debug
    if (key_value > zero_point) // If the key analog value is not at the zero voltage point
    {
#ifdef debug1
      Serial.print("Key Value: ");
      Serial.println(key_value);
#endif

    }

#endif

    do
    {
      // Wait until the key is released
    } while (analogRead(keypad_pin) > zero_point);  // Stay in the loop as long as the keypad is pressed

    decode_key(); // Decode the key press
  }
}

void decode_key() // Function to translate key code to its position number
{
  key_index = 1;  // Local variable for the pointer to the keypad_map array

#ifdef debug1
  Serial.println("Scanning Array");
#endif

  for (int x = 0; x <= 15; x++)  // Scan the keypad_map array looking for closest valid match
  {
#ifdef debug1
    Serial.print("Index: ");
    Serial.print (x);
#endif

    if (abs(key_value - keypad_map[x]) < variance)  // If the data matches within the specified tolerance
    {

#ifdef debug1
      Serial.print("   Match");
      Serial.print("   Key Char: ");
      Serial.println(key_text[x]);
#endif

      key_index = x;  // Set the key pointer to the matching key

      break;  // Exit as soon as we have a valid match
    }

#ifdef debug1
    Serial.println();
#endif

  }
#ifdef debug1
  Serial.print("Key Index: ");
  Serial.println(key_index);
#endif
}

void read_adc() // Function to read the ADS 1115 A/D Converter
{
  adc_value = ads.readADC_SingleEnded(0); // Get the raw ADC value

#ifdef debug3
  Serial.print("Current ADC Value: ");
  Serial.println(adc_value);
#endif

  // Now map the value to degrees (-180 to +180)
  actual_bearing = map(adc_value, adc_cal_0, adc_cal_max, -180, 180); // This is the true bearing (-180 to + 180 degrees)
  // Only process the data if it's outside of the degree tolerance value or if it's the first time through this code
  if ((abs(actual_bearing - previous_bearing) > degree_tolerance) || previous_bearing == -1)
  {
    // Convert the bearing from -180 to +180 into two zones of 180 - 359 and 0 - 180 for proper calculation and display
    if (actual_bearing < -180)  // Set the lower limit to -180
    {
      actual_bearing = -180;
    }

    if (actual_bearing > 180) // Set the upper limit to +180
    {
      actual_bearing = 180;
    }

#ifdef debug2
    Serial.print("Actual Bearing: ");
    Serial.print(actual_bearing);
#endif

    if (actual_bearing < 0)    // Now map it to the display value
    {
      current_bearing = actual_bearing + 360;
    } else {
      current_bearing = actual_bearing;
    }

#ifdef debug2
    Serial.print("   Current Bearing: ");
    Serial.println(current_bearing);
#endif

    previous_bearing = actual_bearing;  // Update the previous bearing value
    data_changed = true;  // Indicate the display needs to be updated
  }
}

void update_display() // Updates the TFT display
{
  if (first_pass) // Only do this part the first time the function is called
  {
    // Clear the screen and display normal operation
    tft.fillScreen(ST7735_BLACK); // Clear the display
    delay(tft_delay);
    tft.setTextSize(1); // Set the text size to 1
    delay(tft_delay);
    tft.setTextColor(ST7735_BLUE); // Set the text color to BLUE
    delay(tft_delay);
    tft.setCursor(35, 5);
    delay(tft_delay);
    tft.print("Current Bearing");  // Display Current Rotator Bearing
    tft.setCursor(35, 60);
    delay(tft_delay);
    tft.print("Desired Bearing");  // Display Entered Rotator Bearing
    delay(tft_delay);
    tft.setCursor(67, 115);
    delay(tft_delay);
    tft.setTextColor(ST7735_GREEN); // Set the text color to GREEN
    delay(tft_delay);
    tft.print("Ready");
    first_pass = false; // Turn off the first pass flag
  } else
  {
    clear_data();  // Clear the value display area

    tft.setTextSize(4); // Set the text size to 4
    delay(tft_delay);
    if (moving) // If the rotator is turning set the current bearing color to yellow - otherwise set it to green
    {
      tft.setTextColor(ST7735_YELLOW); // Set the text color to GREEN
      delay(tft_delay);
    } else
    {
      tft.setTextColor(ST7735_GREEN); // Set the text color to GREEN
      delay(tft_delay);
    }
    tft.setCursor(50, 25);
    delay(tft_delay);

    convert_bearing(current_bearing); // convert the current bearing to a 3 digit formatted string
    tft.setTextColor(ST7735_GREEN); // Set the text color to GREEN
    delay(tft_delay);
    if (entered_bearing != -1)  // If the entered bearing is a valid entry
    {
      tft.setCursor(50, 80);
      delay(tft_delay);
      convert_bearing(entered_bearing);// Convert the entered bearing to a 3 digit formatted string
    }
    data_changed = false; // The display is updated - Turn off the display update flag
  }

  next_update_time = millis() + update_time;  // Calculate the time for the next update

}

void clear_data() //Clears the data area of the display
{
  tft.fillRect(45, 23, 78, 33, ST7735_BLACK); // Clear the display data areas
  delay(tft_delay);
  tft.fillRect(45, 78, 78, 33, ST7735_BLACK); // Clear the display data areas
  delay(tft_delay);
}

void clear_status() // Clears the status line of the display
{
  tft.fillRect(0, 112, 159, 15, ST7735_BLACK); // Clear the display data areas
  delay(tft_delay);
}

void start_rotate() // Handles the rotate process
{
  // sets up and starts the rotation process

  // Check to be sure it's a valid move (greater than degree tolerance) Move has already been checked for being a valid entry

  // We use actual bearing value since we may cross the 359 to 0 degree boundary
  // We already have actual - we need to calculate it for the desired destination

  if (entered_bearing > 180)  // Convert the destination to the -180 to +180 actual degree format
  {
    // We're in 181 to 359 area - subtract 180 degrees
    destination_bearing = entered_bearing - 360;
  } else
  {
    // We're in 0 to 180 - don't do anything with it
    destination_bearing = entered_bearing;
  }

#ifdef debug4
  Serial.print("Current = ");
  Serial.print(actual_bearing);
  Serial.print("   Start Moving - Destination = ");
  Serial.print(entered_bearing);
  Serial.print("  Actual Destination Bearing = ");
  Serial.print(destination_bearing);
#endif

  // Only start if the desired destination is outside of the degree tolerance
  if (abs(actual_bearing - destination_bearing) > degree_tolerance)
  {
    if (destination_bearing == 180) // If desired bearing is 180 degrees, turn towards the closest 180 degree point
    {
      // Find the nearest one and go to it
      if (actual_bearing < 0) // Convert the destination bearing to the actual bearing format
      {
        // Turn Left
        destination_bearing = -180;

#ifdef debug4
        Serial.println("  Turn left");
#endif

        rotate_direction = "L"; // The rotation direction to the nearest 180 should be Left
        rotate(); // Start the rotation
      } else
      {
        // Turn Right

#ifdef debug4
        Serial.println("  Turn right");
#endif
        rotate_direction = "R"; // The rotation direction to the nearest 180 should be Right
        rotate(); // Start the rotation
      }
    } else
    {
      // Do we turn right (clockwise) or left (counterclockwise)?
      if (actual_bearing > destination_bearing) //  If the current bearing is greater than the destination - turn left
      {
        // Turn left

#ifdef debug4
        Serial.println("  Turn left");
#endif

        rotate_direction = "L"; //  The rotation direction should be Left
        rotate(); // Start the rotation

      } else
      {
        // Turn right
#ifdef debug4
        Serial.println("  Turn right");
#endif

        rotate_direction = "R"; //  The rotation direction should be Right
        rotate(); // Start the rotation
      }

      moving = true;  // Set the rotating flag
    }
  } else
  {
    // Don't execute the move
    moving = false; // Turn off the moving flag
    move_to = ""; // Reset the desired bearing string
    entered_bearing = -1; // Reset the entered bearing value
    data_changed = true;  // Indicate that the display needs to be updated
  }
}

void convert_bearing(int calc_bearing)  // Function to convert int bearing to String
{
  // Convert the bearing to a 3 character string and pad with zeroes
  String display_bearing = String(calc_bearing);
  while (display_bearing.length() < 3)    //  Do this while the string length is less than 3 digits
  {
    // While the string length is less than 3 digits
    display_bearing = "0" + display_bearing;  // Pad with leading zeroes until we have a 3 digit value
  }
  tft.print(display_bearing); // Display the padded bearing value
}

void stop_all() // Function to stop all rotation
{
  // Stop all rotation and trigger braking delay
#ifdef debug4
  Serial.println("Stop all Rotation - Initiate Braking delay");
#endif

  clear_status(); // Clear the display status line
  tft.setTextSize(1); // Set the text size to 1
  tft.setTextColor(ST7735_RED); // Set the text color to RED
  delay(tft_delay);
  tft.setCursor(65, 115);
  delay(tft_delay);
  tft.print("Braking"); // Indicate we are braking
  digitalWrite(left_relay, LOW);
  digitalWrite(right_relay, LOW);
  delay(brake_time);
  digitalWrite(brake_relay, LOW);
  clear_status();
  tft.setTextColor(ST7735_GREEN); // Set the text color to GREEN
  delay(tft_delay);
  tft.setCursor(67, 115);
  delay(tft_delay);
  tft.print("Ready");

  moving = false; // Turn off the moving flag
  move_to = ""; // Reset the desired bearing string
  entered_bearing = -1; // Reset the entered bearing value
  data_changed = true;  // Indicate that the display needs to be updated
}

void rotate() // Function to perform rotation
{
  clear_status(); // Clear the display status line
  tft.setTextSize(1); // Set the text size to 1
  delay(tft_delay);
  tft.setTextColor(ST7735_YELLOW); // Set the text color to YELLOW
  delay(tft_delay);
  tft.setCursor(60, 115);
  delay(tft_delay);
  tft.print("Rotating"); // Indicate we are rotating
  if (rotate_direction == "L")  // Turn on the left rotation relay and release the brake
  {
    digitalWrite(brake_relay, HIGH);  // Release the brake
    delay(release_delay);
    digitalWrite(left_relay, HIGH); // Enable the left relay
  } else
  {
    // Turn on the left rotation relay and release the brake
    digitalWrite(brake_relay, HIGH);  // Release the brake
    delay(release_delay);
    digitalWrite(right_relay, HIGH); // Enable the right relay
  }
}

void check_move()
{
  // We only get here if it's already moving
  // Checks to see if we've reached the destination bearing
  // We've just read the ADC so no need to read it again
  if (rotate_direction == "L")  // destination is less than current position
  {
    if (actual_bearing <= (destination_bearing + left_rotate_trim))  // Stop when we get there - include left rotate trim
    {
      stop_all(); // Stop all rotation
    }
  } else
  {
    // We're turning right
    if (actual_bearing >= (destination_bearing + right_rotate_trim))   // Stop when we get there - include right rotate trim
    {
      stop_all(); // Stop all rotation
    }
  }
}

